<?php
/**
 * Exit if accessed directly
 */
if ( !defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Check if WooCommerce is active
 */
include_once( ABSPATH . 'wp-admin/includes/plugin.php' );
if ( is_plugin_active( 'woocommerce/woocommerce.php' ) ) {

	/**
	 * Init plugin
	 */
	function woocommerce_dpdfrance_predict_init() {
		if ( ! class_exists( 'WC_dpdfrance_predict' ) ) {

			class WC_dpdfrance_predict extends WC_Shipping_Method {
				/**
				 * @var string
				 */
				private $table_rate_option;
				/**
				 * @var string
				 */
				private $last_table_rate_id_option;
				/**
				 * @var string
				 */
				private $zones_option;
				/**
				 * @var string
				 */
				private $last_zone_id_option;
				/**
				 * @var array
				 */
				private $zones;
				/**
				 * @var integer
				 */
				private $last_zone_id;
				/**
				 * @var array
				 */
				private $table_rates;
				/**
				 * @var integer
				 */
				private $last_table_rate_id;

				/**
				 * WC_dpdfrance_predict constructor.
				 */
				public function __construct() {
					parent::__construct();
					$this->id                        = 'dpdfrance_predict';
					$this->table_rate_option         = $this->id . '_table_rates';          // ? dpdfrance_predict_table_rate_option          | Nom afin de récupérer une option dans la classe
					$this->last_table_rate_id_option = $this->id . '_last_table_rate_id';   // ? dpdfrance_predict_last_table_rate_id_option  | Nom afin de récupérer une option dans la classe
					$this->zones_option              = $this->id . '_zones';                // ? dpdfrance_predict_zones_option               | Nom afin de récupérer une option dans la classe
					$this->last_zone_id_option       = $this->id . '_last_zone_id';         // ? dpdfrance_predict_last_zone_id_option        | Nom afin de récupérer une option dans la classe
					$this->method_title              = __( 'Predict by DPD', 'woocommerce-dpdfrance' );
					$this->method_description        = __( 'Predict by DPD delivery plugin for WooCommerce - France vendors only', 'woocommerce-dpdfrance' );
					$this->zones                     = [];
					$this->last_zone_id              = null;
					$this->last_table_rate_id        = null;
					$this->table_rates               = [];
					$this->get_zones();
					$this->get_last_zone_id();
					$this->get_table_rates();
					$this->get_last_table_rate_id();
					$this->init();
				}

				/**
				 * * Retrieves zones array from database
				 */
				public function get_zones() {
					$this->zones = array_filter( (array) get_option( $this->zones_option ) );
				}

				/**
				 * * Retrieves last zone id from database
				 */
				public function get_last_zone_id() {
					$this->last_zone_id = (int) get_option( $this->last_zone_id_option );
				}

				/**
				 * * Retrieves table_rates array from database
				 */
				public function get_table_rates() {
					$this->table_rates = array_filter( (array) get_option( $this->table_rate_option ) );
				}

				/**
				 * * Retrieves last table_rate id from database
				 */
				public function get_last_table_rate_id() {
					$this->last_table_rate_id = (int) get_option( $this->last_table_rate_id_option );
				}

				/**
				 * * [BO][WooCommerce][Réglages][Expedition][DPD Predict] init shipping method settings
				 */
				public function init() {
					// Load the settings API
					$this->init_form_fields(); // This is part of the settings API. Override the method to add your own settings
					$this->init_settings();    // This is part of the settings API. Loads settings you previously init.

					$this->enabled = $this->settings['enabled'];
					$this->title   = $this->settings['title']; // Shown in drop down and admin order screen

					// Save settings in admin if you have any defined
					add_action( 'woocommerce_update_options_shipping_' . $this->id, [ $this, 'process_admin_options' ] );
					add_action( 'woocommerce_update_options_shipping_' . $this->id, [ $this, 'admin_update_submitted_shipping_zone' ] );
					add_action( 'woocommerce_update_options_shipping_' . $this->id, [ $this, 'admin_update_submitted_shipping_rates' ] );
				}

				/**
				 * Initialise Predict Settings Form Fields.
				 */
				public function init_form_fields() {
					$this->form_fields = WC_DPDFrance_Settings_Predict::settings_predict();
				}

				/**
				 * Display settings forms based of the settings array
				 */
				public function admin_options() {
					?>
                    <script type="application/javascript">
                        let options = <?= json_encode( WC_DPDFrance_Tools::admin_prepare_dropdown_options( $this->zones ) ) ?>;
                    </script>
                    <script type="text/javascript" src="<?= OS_DPDFRANCE_ROOT_URL ?>/assets/js/back/dpd_shipping_setting.js"></script>
                    <h2><?php _e( 'Predict by DPD', 'woocommerce-dpdfrance' ); ?></h2>
                    <div class="updated woocommerce-message">
                        <p>
                            <strong><?php _e( 'Welcome to DPD.', 'woocommerce-dpdfrance' ); ?></strong> <?php _e( 'The plugin documentation is available here :', 'woocommerce-dpdfrance' ); ?>
                            <a target="_blank" href="<?php echo OS_DPDFRANCE_ROOT_URL; ?>/assets/docs/readme_dpdfrance_woocommerce.pdf">
                                <img src="<?php echo OS_DPDFRANCE_ROOT_URL; ?>/assets/img/admin/pdf.png" alt="instructions pdf"/>
                            </a>
                        </p>
                    </div>
                    <table class="form-table">
						<?php $this->generate_settings_html(); ?>
                    </table>

					<?php
				}

				/**
				 * * Generates HTML code for zone settings table
				 */
				public function generate_zones_table_html() {
					ob_start();
					?>
                    <tr valign="top">
                        <th scope="row" class="titledesc"><?php _e( 'Shipping Zones', 'woocommerce-dpdfrance' ); ?></th>
                        <td class="forminp" id="<?php echo $this->id; ?>_zones">
                            <p style="padding-bottom: 10px;">
								<?php _e( 'After adding a shipping zone, hit "Save changes" so that it appears as an option in the table rate section.', 'woocommerce-dpdfrance' ); ?>
                            </p>
                            <table class="shippingrows widefat" cellspacing="0">
                                <col style="width:0%">
                                <col style="width:0%">
                                <col style="width:100%;">
                                <thead>
                                <tr>
                                    <th class="check-column"><input type="checkbox"></th>
                                    <th>
										<?php _e( 'Name', 'woocommerce-dpdfrance' ); ?>
                                        <a class="tips" data-tip="<?php _e( 'Shipping zone name, will appear in table rates table.', 'woocommerce-dpdfrance' ); ?>">[?]</a>
                                    </th>
                                    <th>
										<?php _e( 'Countries', 'woocommerce-dpdfrance' ); ?>
                                        <a class="tips" data-tip="<?php _e( 'Add one or more countries that are part of this shipping zone.', 'woocommerce-dpdfrance' ); ?>">[?]</a>
                                    </th>
                                </tr>
                                </thead>
                                <tbody class="zones">
                                    <tr class="zone">
                                        <th></th>
                                        <td>
                                            <div style="width: 200px;"><?php _e( 'Default Zone (everywhere else)', 'woocommerce-dpdfrance' ); ?></div>
                                        </td>
                                        <td>
                                            <em><?php _e( 'All allowed countries', 'woocommerce-dpdfrance' ); ?></em>
                                        </td>
                                    </tr>
                                </tbody>
                                <tfoot>
                                    <tr>
                                        <th colspan="3">
                                            <a href="#" class="add button"><?php _e( 'Add Shipping Zone', 'woocommerce-dpdfrance' ); ?></a>
                                            <a href="#" class="remove button"><?php _e( 'Delete Selected Zones', 'woocommerce-dpdfrance' ); ?></a>
                                        </th>
                                    </tr>
                                </tfoot>
                            </table>
                            <script type="text/javascript">
                                /* ? Id de la derniere zone existante */
                                let lastZoneId   = <?= $this->last_zone_id ?>;
                                /* ? Les zones deja existante */
                                let tables_zones = <?= json_encode( $this->zones ) ?>;
                                // ? injection des zones d'expéditions sauvegardées dans la BDD
                                printSavedDataInTable(tables_zones, '#dpdfrance_predict_zones tbody.zones', 'zone', "<?= $this->id ?>")

                                jQuery(function () {
                                    /* ? Add Row To Zone*/
                                    jQuery('#<?= $this->id ?>_zones').on('click', 'a.add', function () {
                                        jQuery('#<?= $this->id ?>_zones table tbody tr:last').after(addZoneRowHtml(true, false, "<?= $this->id ?>"));
                                        /* ? Si j'ai la librairie jquery Chosen */
                                        if (jQuery().chosen) {
                                            jQuery("select.chosen_select").chosen({
                                                width                   : '350px',
                                                disable_search_threshold: 5
                                            });
                                        } else {
                                            /* ? Sinon utilisation de la librairie SelectWoo fork Select2() */
                                            jQuery("select.chosen_select").selectWoo();
                                        }

                                        return false;
                                    });

                                    /* ? Remove Row To Zone*/
                                    jQuery('#<?php echo $this->id; ?>_zones').on('click', 'a.remove', function () {
                                        let answer = confirm("<?php _e( 'Delete the selected zones?', 'woocommerce-dpdfrance' ); ?>");
                                        if (answer) {
                                            jQuery('#<?php echo $this->id; ?>_zones table tbody tr th.check-column input:checked').each(function (i, el) {
                                                jQuery(el).closest('tr').remove();
                                            });
                                        }
                                        return false;
                                    });

                                });
                            </script>
                        </td>
                    </tr>
					<?php
					return ob_get_clean();
				}

				/**
                 * Generates HTML for table_rate settings table
				 */
				public function generate_table_rates_table_html() {
					ob_start();
					?>
                    <tr valign="top">
                        <th scope="row" class="titledesc"><?php _e( 'Table Rates', 'woocommerce-dpdfrance' ); ?></th>
                        <td class="forminp" id="<?php echo $this->id; ?>_table_rates">
                            <table class="shippingrows widefat" cellspacing="0">
                                <col style="width:0%">
                                <col style="width:0%">
                                <col style="width:0%">
                                <col style="width:0%">
                                <col style="width:0%">
                                <col style="width:100%;">
                                <thead>
                                    <tr>
                                        <th class="check-column"><input type="checkbox"></th>
                                        <th class="debug-col"><?php _e( 'ID', 'woocommerce-dpdfrance' ); ?></th>
                                        <th>
								    		<?php _e( 'Zone', 'woocommerce-dpdfrance' ); ?>
                                            <a class="tips" data-tip="<?php _e( 'Shipping zone, as defined in Shipping Zones table.', 'woocommerce-dpdfrance' ); ?>">[?]</a>
                                        </th>
                                        <th>
								    		<?php _e( 'Condition', 'woocommerce-dpdfrance' ); ?>
                                            <a class="tips" data-tip="<?php _e( 'Choose which metric to base your table rate on.', 'woocommerce-dpdfrance' ); ?>">[?]</a>
                                        </th>
                                        <th>
								    		<?php _e( 'Min', 'woocommerce-dpdfrance' ); ?>
                                            <a class="tips" data-tip="<?php _e( 'Minimum, in decimal format. Inclusive.', 'woocommerce-dpdfrance' ); ?>">[?]</a>
                                        </th>
                                        <th>
								    		<?php _e( 'Max', 'woocommerce-dpdfrance' ); ?>
                                            <a class="tips" data-tip="<?php _e( 'Maximum, in decimal format. Inclusive. To impose no upper limit, use *".', 'woocommerce-dpdfrance' ); ?>">[?]</a>
                                        </th>
                                        <th>
								    		<?php _e( 'Cost', 'woocommerce-dpdfrance' ); ?>
                                            <a class="tips" data-tip="<?php _e( 'Cost, excluding tax.', 'woocommerce-dpdfrance' ); ?>">[?]</a>
                                        </th>
                                    </tr>
                                </thead>
                                <tbody class="table_rates">
                                </tbody>
                                <tfoot>
                                    <tr>
                                        <th colspan="7">
                                            <a href="#" class="add button"><?php _e( 'Add Table Rate', 'woocommerce-dpdfrance' ); ?></a>
                                            <a href="#" class="remove button"><?php _e( 'Delete Selected Rates', 'woocommerce-dpdfrance' ); ?></a>
                                        </th>
                                    </tr>
                                </tfoot>
                            </table>
                            <script type="application/javascript">

                                /* ? Id du dernier tarif existant*/
                                let lastTableRateId = <?= $this->last_table_rate_id ?>;
                                /* ? Les tarifs deja existant*/
                                let tables_rates    = <?= json_encode( $this->table_rates ) ?>;
                                /* ? Zone par defaut */
                                let zone            = '<?= ( ! empty( $this->zones[0]['id'] ) ) ? $this->zones[0]['id'] : 0 ?>';
                                // ? injection des prix de frais de port sauvegardé dans la BDD
                                printSavedDataInTable(tables_rates, '.table_rates', 'rate', "<?= $this->id ?>", zone);

                                jQuery(function () {
                                    /* ? Add Row To Rate*/
                                    jQuery('#<?= $this->id ?>_table_rates').on('click', 'a.add', function () {
                                        jQuery(addTableRateRowHtml(true, false, '<?= $this->id ?>', zone)).appendTo('#<?= $this->id ?>_table_rates table tbody');
                                        return false;
                                    });

                                    /* ? Remove Row To Rate*/
                                    jQuery('#<?= $this->id ?>_table_rates').on('click', 'a.remove', function () {
                                        let answer = confirm("<?php _e( 'Delete the selected rates?', 'woocommerce-dpdfrance' ); ?>");
                                        if (answer) {
                                            jQuery('#<?= $this->id ?>_table_rates table tbody tr th.check-column input:checked').each(function (i, el) {
                                                jQuery(el).closest('tr').remove();
                                            });
                                        }
                                        return false;
                                    });

                                });
                            </script>
                        </td>
                    </tr>
					<?php
					return ob_get_clean();
				}

				/**
                 * * Process and update submitted shipping zones
				 */
				public function admin_update_submitted_shipping_zone() {

					// Save the zones
					$zone_id = $zone_name = $zone_country = $zone_include = $zone_exclude = $zone_type = $zone_enabled = $zones = [];

					if ( isset( $_POST[$this->id . '_zone_id'] ) ) {
						$zone_id = array_map( 'wc_clean', $_POST[$this->id . '_zone_id'] );
					}
					if ( isset( $_POST[$this->id . '_zone_name'] ) ) {
						$zone_name = array_map( 'wc_clean', $_POST[$this->id . '_zone_name'] );
					}
					if ( isset( $_POST[$this->id . '_zone_country'] ) ) {
						$zone_country = $_POST[$this->id . '_zone_country'];
					}
					if ( isset( $_POST[$this->id . '_zone_include'] ) ) {
						$zone_include = array_map( 'wc_clean', $_POST[$this->id . '_zone_include'] );
					}
					if ( isset( $_POST[$this->id . '_zone_exclude'] ) ) {
						$zone_exclude = array_map( 'wc_clean', $_POST[$this->id . '_zone_exclude'] );
					}
					if ( isset( $_POST[$this->id . '_zone_type'] ) ) {
						$zone_type = array_map( 'wc_clean', $_POST[$this->id . '_zone_type'] );
					}
					if ( isset( $_POST[$this->id . '_zone_enabled'] ) ) {
						$zone_enabled = array_map( 'wc_clean', $_POST[$this->id . '_zone_enabled'] );
					}

					// Get max key
					$values = $zone_id;
					ksort( $values );
					$value = end( $values );
					$key   = key( $values );

					for ( $i = 0; $i <= $key; $i++ ) {
						if (
						     isset( $zone_id[ $i ], $zone_include[ $i ], $zone_exclude[ $i ], $zone_type[ $i ], $zone_enabled[ $i ] ) &&
                             !empty( $zone_name[ $i ] ) && !empty( $zone_country[ $i ] )
                        ) {

							// Add to zones array
							$zones[] = [
								'id'      => $zone_id[ $i ],
								'name'    => $zone_name[ $i ],
								'country' => $zone_country[ $i ],
								'include' => $zone_include[ $i ],
								'exclude' => $zone_exclude[ $i ],
								'type'    => $zone_type[ $i ],
								'enabled' => $zone_enabled[ $i ],
							];
						}
					}

					if (
					        ( !empty( $zone_id[ $key ] ) ) &&
                            ( $zone_id[ $key ] > $this->last_zone_id ) &&
                            ( is_numeric( $zone_id[ $key ] ) )
                    ) {
						$highest_zone_id = $zone_id[ $key ];
						update_option( $this->last_zone_id_option, $highest_zone_id );
					}

					update_option( $this->zones_option, $zones );

					$this->get_zones();

				}

				/**
				 * * Process and update submitted table_rates
				 */
				public function admin_update_submitted_shipping_rates() {
					// Save the rates
					$table_rate_id = $table_rate_zone = $table_rate_basis = $table_rate_min = $table_rate_max = $table_rate_cost = $table_rate_enabled = $table_rates = [];

					if ( isset( $_POST[$this->id . '_table_rate_id'] ) ) {
						$table_rate_id = array_map( 'wc_clean', $_POST[$this->id . '_table_rate_id'] );
					}
					if ( isset( $_POST[$this->id . '_table_rate_zone'] ) ) {
						$table_rate_zone = array_map( 'wc_clean', $_POST[$this->id . '_table_rate_zone'] );
					}
					if ( isset( $_POST[$this->id . '_table_rate_basis'] ) ) {
						$table_rate_basis = array_map( 'wc_clean', $_POST[$this->id . '_table_rate_basis'] );
					}
					if ( isset( $_POST[$this->id . '_table_rate_min'] ) ) {
						$table_rate_min = array_map( 'stripslashes', $_POST[$this->id . '_table_rate_min'] );
					}
					if ( isset( $_POST[$this->id . '_table_rate_max'] ) ) {
						$table_rate_max = array_map( 'stripslashes', $_POST[$this->id . '_table_rate_max'] );
					}
					if ( isset( $_POST[$this->id . '_table_rate_cost'] ) ) {
						$table_rate_cost = array_map( 'stripslashes', $_POST[$this->id . '_table_rate_cost'] );
					}
					if ( isset( $_POST[$this->id . '_table_rate_enabled'] ) ) {
						$table_rate_enabled = array_map( 'wc_clean', $_POST[$this->id . '_table_rate_enabled'] );
					}

					// Get max key
					$values = $table_rate_id;
					ksort( $values );
					$value = end( $values );
					$key   = key( $values );

					for ( $i = 0; $i <= $key; $i++ ) {
						if ( isset( $table_rate_id[ $i ], $table_rate_zone[ $i ], $table_rate_basis[ $i ], $table_rate_min[ $i ], $table_rate_max[ $i ], $table_rate_cost[ $i ], $table_rate_enabled[ $i ] ) ) {

							$table_rate_cost[ $i ] = wc_format_decimal( $table_rate_cost[ $i ] );
							// Add table_rates to array
							$table_rates[]         = [
								'id'      => $table_rate_id[ $i ],
								'zone'    => $table_rate_zone[ $i ],
								'basis'   => $table_rate_basis[ $i ],
								'min'     => $table_rate_min[ $i ],
								'max'     => $table_rate_max[ $i ],
								'cost'    => $table_rate_cost[ $i ],
								'enabled' => $table_rate_enabled[ $i ],
							];
						}
					}

					if (
					     ( !empty( $table_rate_id[ $key ] ) ) &&
                         ( $table_rate_id[ $key ] > $this->last_table_rate_id ) &&
                         ( is_numeric( $table_rate_id[ $key ] ) )
					) {
						$highest_table_rate_id = $table_rate_id[ $key ];
						update_option( $this->last_table_rate_id_option, $highest_table_rate_id );
					}

					update_option( $this->table_rate_option, $table_rates );

					$this->get_table_rates();

				}

				/**
                 * Calculate shipping cost. This is called by WooCommerce
				 * @param array $package
				 */
				public function calculate_shipping( $package = [] ) {

					$available_table_rates = WC_DPDFrance_Tools::get_available_table_rates( $package, $this->zones, $this->table_rates );
					$table_rate            = WC_DPDFrance_Tools::pick_cheapest_table_rate( $available_table_rates );
					$tax                   = ! isset( $this->settings['tax_status'] ) || $this->settings['tax_status'] === 'none' ? false : '';

					if ( $table_rate !== false ) {

						$cost                 = WC_DPDFrance_Tools::check_cart_has_handle_fee( $this->settings, $table_rate );
						$cart_context         = WC_DPDFrance_Tools::get_order_shipping_cost( $this->id, $package['destination']['postcode'], $cost );
						$cart_context['cost'] = WC_DPDFrance_Tools::check_cart_can_use_coupons( $this->settings, $cart_context['cost'] );
						$cart_context['cost'] = WC_DPDFrance_Tools::check_cart_can_have_free_shipping( $this->settings, $cart_context['cost'] );
						$cart_context['excl'] = WC_DPDFrance_Tools::check_cart_is_overweighted( $package, $this->settings, $cart_context['excl'] );
						$cart_context['excl'] = WC_DPDFrance_Tools::check_cart_has_excluded_products( $package, $this->settings, $cart_context['excl'] );

						/* Register the rate */
						$rate = [
							'id'       => $this->id,
							'label'    => $this->title . ( isset( $cart_context['cost'] ) && $cart_context['cost'] === 0 ? ': ' . __( 'free', 'woocommerce-dpdfrance' ) : '' ),
							'cost'     => $cart_context['cost'] ?? null,
							'taxes'    => $tax,
							'calc_tax' => 'per_order',
						];

						if (
							isset( $cart_context['cost'] ) &&
							$cart_context['excl'] === 0
						) {
							$this->add_rate( $rate );
						}
					}
				}
			}
		}
		/*--------------------------------INSERT SHIPPING METHOD------------------------------------------------------*/
		/**
		 * * [BO][WooCommerce][Réglages][Expedition][DPD Predict] Add shipping method
		 */
		function add_dpdfrance_predict_delivery_method( $methods ) {
			$methods[] = 'WC_dpdfrance_predict';
			return $methods;
		}

		/**
		 * * [BO][WooCommerce][Réglages][Expedition][DPD Predict] Hide shipping cost 0.00 when free shipping
		 */
		add_filter( 'woocommerce_cart_shipping_method_full_label', function ( $label, $method ) {
			if ( $method->id === 'dpdfrance_predict' && $method->cost <= 0 ) {
				$label = $method->get_label();
			}
			return $label;
		}, 10, 2 );

		add_filter( 'woocommerce_shipping_methods', 'add_dpdfrance_predict_delivery_method' );
	}

	/*--------------------------------TRADUCTION----------------------------------------------------------------------*/
	/**
     * * Load Translation files
     */
	function woocommerce_dpdfrance_predict_load_plugin_textdomain() {
		load_plugin_textdomain( 'woocommerce-dpdfrance', false, basename( __DIR__ ) . '/../../../languages/plugins/' );
	}

	/**
	 * * Fires after WordPress has finished loading but before any headers are sent.
	 */
	add_action( 'init', 'woocommerce_dpdfrance_predict_load_plugin_textdomain' );

	/*--------------------------------LOAD ASSETS IN CHECKOUT PAGE----------------------------------------------------*/
	/**
	 * * Loads custom JS file
	 */
	function woocommerce_dpdfrance_predict_load_assets() {
		wp_enqueue_script(
			'dpdfrance_predict_custom_script',
			OS_DPDFRANCE_ROOT_URL . '/assets/js/front/predict/dpdfrance_predict.js',
			[ 'jquery' ]
		);
		wp_localize_script(
			'dpdfrance_predict_custom_script',
			'objectL10n',
			[
				'gsm_ok' => __( 'Your Predict delivery by DPD : Your mobile phone number is correct, you can proceed to payment', 'woocommerce-dpdfrance' ),
				'gsm_ko' => __( 'Your Predict delivery by DPD : Before payment, please provide a correct mobile phone number starting with the international prefix (i.e. +33)', 'woocommerce-dpdfrance' ),
			]
		);
		wp_enqueue_style(
			'dpdfrance_predict_custom_css',
			OS_DPDFRANCE_ROOT_URL . '/assets/css/front/dpdfrance_shipping.css'
		);
	}

	/**
	 * * Fires after WordPress has finished loading but before any headers are sent.
	 */
	add_action( 'woocommerce_checkout_shipping', 'woocommerce_dpdfrance_predict_load_assets' );

    /*--------------------------------IN CHECKOUT PAGE - DISPLAY PREDICT INFOS----------------------------------------*/
    /**
     * * [FO] Checks if selected carrier is dpdfrance_predict, and controls value of shipping_company field
     */
    function woocommerce_dpdfrance_predict_checkout_controller()
    {
        global $woocommerce;
        $dpdpredict_settings = get_option( 'woocommerce_dpdfrance_predict_settings' );

        if ($dpdpredict_settings['enabled'] !== 'no') {
            $my_WC_dpdfrance_predict = new WC_dpdfrance_predict();
            @parse_str($_POST['post_data'], $post_data);

            $postcode = @($post_data['ship_to_different_address'] === 1 ? $post_data['shipping_postcode'] : $post_data['billing_postcode']);
            $cart_context = WC_DPDFrance_Tools::get_order_shipping_cost('dpdfrance_predict', $postcode);

            if ($woocommerce->session->chosen_shipping_methods[0] === 'dpdfrance_predict') {
                print_r(
                    '   <tr>
                            <td colspan="2">
                                <div id="div_dpdfrance_predict_header">
                                    <p>' . __('Your Predict delivery by DPD', 'woocommerce-dpdfrance') . '</p>
                                </div>
                                <div class="module" id="predict">
                                    <div id="div_dpdfrance_predict_logo"></div>
					            		<div class="copy">
                                            <p></p>
                                            <h2>' . __('With Predict, be independent with your deliveries !', 'woocommerce-dpdfrance') . '</h2>
                                            <p></p><br/>
					            			<p></p>
					            			<h2>' . __('How it works:', 'woocommerce-dpdfrance') . '</h2>
					            			<p></p>
					            			<ul>
                                                <li><b>' . __('A first SMS will be sent to you when your parcel is taken in charge by DPD France. This SMS will also tell you the expected delivery date.', 'woocommerce-dpdfrance') . '</b></li>
                                                <li><b>' . __('On the day of delivery, you will receive an SMS and an email indicating the delivery time slot and your secure 4-digit code. This code will allow you to receive your order in complete security.', 'woocommerce-dpdfrance') . '</b></li>
                                                <li><b>' . __('Upon delivery, transmit your code to the DPD France delivery person. Your parcel will be delivered to you against signature.', 'woocommerce-dpdfrance') . '</b></li>
                                            </ul><br/>
					            			<p>' . __('The Predict service allows you to be autonomous on your delivery thanks to a wide choice of reprogramming options until the day before or in case of absence :', 'woocommerce-dpdfrance') . '</p>
                                            <ul>
                                                <li><b>' . __('Delivery of your parcel at pudo’s', 'woocommerce-dpdfrance') . '</b></li>
                                                <li><b>' . __('Delivery to your workplace', 'woocommerce-dpdfrance') . '</b></li>
                                                <li><b>' . __('Delivery to another address', 'woocommerce-dpdfrance') . '</b></li>
					            				<li><b>' . __('Delivery to a neighbour', 'woocommerce-dpdfrance') . '</b></li>
                                                <li><b>' . __('Withdrawal in a DPD France depot', 'woocommerce-dpdfrance') . '</b></li>
                                                <li><b>' . __('Delivery to another date', 'woocommerce-dpdfrance') . '</b></li>
                                            </ul>
					            			<p>' . __('Benefit from the advantages of Predict delivery !', 'woocommerce-dpdfrance') . '</p>
                                        </div><br/>
                                    <div id="div_dpdfrance_dpd_logo"></div>
                                </div>
                                <div id="div_dpdfrance_predict_gsm">
                                    ' . __('To get all the advantages of DPD\'s Predict service, please provide a french mobile phone number', 'woocommerce-dpdfrance') . '
                                </div>
                            </td>
                        </tr>
                        <tr class="gsmValidity">
                        </tr>'
                );
            }
        }
    }

    add_action('woocommerce_review_order_after_shipping', 'woocommerce_dpdfrance_predict_checkout_controller');
}
