<?php

/* * Exit if accessed directly */
if ( !defined( 'ABSPATH' ) ) {
	exit;
}

/* * Check if WooCommerce is active */
include_once( ABSPATH . 'wp-admin/includes/plugin.php' );
/* * Init plugin */
if (
	is_plugin_active( 'woocommerce/woocommerce.php' ) &&
	!class_exists( 'WC_DPDFrance_Tools' )
) {

	/* * Main class */

	class WC_DPDFrance_Tools {

		/**
		 * Return true if WooCommerce version is from 3 and above
		 * @return bool
		 */
		public static function is_from_wc3(): bool {
			return WC()->version >= '3.0.0';
		}

		/**
		 * Get the shipping classes inside cart
		 *
		 * @param $package
		 *
		 * @return array
		 */
		public static function dpdfrance_find_shipping_classes( $package ): array {
			$found_shipping_classes = [];
			foreach ( $package['contents'] as $item_id => $values ) {
				if ( $values['data']->needs_shipping() ) {
					$found_shipping_classes[ $item_id ] = $values['data']->get_shipping_class();
				}
			}

			return $found_shipping_classes;
		}

		/**
		 * Formats GSM numbers
		 *
		 * @param $tel_dest
		 * @param $code_pays_dest
		 *
		 * @return array|false|string|string[]
		 */
		public static function format_gsm( $tel_dest, $code_pays_dest ) {
			$tel_dest = str_replace( [ ' ', '.', '-', ',', ';', '/', '\\', '(', ')' ], '', $tel_dest );
			// Chrome autofill fix
			if ( substr( $tel_dest, 0, 2 ) == 33 ) {
				$tel_dest = substr_replace( $tel_dest, '0', 0, 2 );
			}
			switch ( $code_pays_dest ) {
				case 'F':
					return preg_match( '/^((\+33|0|0033)[67])(?:[ _.-]?(\d{2})){4}$/', $tel_dest ) ? $tel_dest : false;
					break;

				case 'D':
					return preg_match( '/^(\+|00)49(15|16|17)(\s?\d{7,8})$/', $tel_dest ) ? $tel_dest : false;
					break;

				case 'B':
					return preg_match( '/^(\+|00)324([56789]\d)(\s?\d{6,7})$/', $tel_dest ) ? $tel_dest : false;
					break;

				case 'AT':
					return preg_match( '/^(\+|00)436([56789]\d)(\s?\d{4,5})$/', $tel_dest ) ? $tel_dest : false;
					break;

				case 'GB':
					return preg_match( '/^(\+|00)447([3456789]\d)(\s?\d{7,8})$/', $tel_dest ) ? $tel_dest : false;
					break;

				case 'NL':
					return preg_match( '/^(\+|00)316(\s?\d{8,9})$/', $tel_dest ) ? $tel_dest : false;
					break;

				case 'P':
					return preg_match( '/^(\+|00)3519(\s?\d{8,9})$/', $tel_dest ) ? $tel_dest : false;
					break;

				case 'IRL':
					return preg_match( '/^(\+|00)3538(\s?\d{8,9})$/', $tel_dest ) ? $tel_dest : false;
					break;

				case 'E':
					return preg_match( '/^(\+|00)34(6|7)(\s?\d{8,9})$/', $tel_dest ) ? $tel_dest : false;
					break;

				case 'I':
					return preg_match( '/^(\+|00)393(\s?\d{9,10})$/', $tel_dest ) ? $tel_dest : false;
					break;

				default:
					return $tel_dest;
					break;
			}
		}

		/**
		 * Strip Accent - Format string for Cargo
		 *
		 * @param $str
		 *
		 * @return array|string|string[]|null
		 */
		public static function strip_accents( $str ) {
			$str = preg_replace( '/[\x{00C0}\x{00C1}\x{00C2}\x{00C3}\x{00C4}\x{00C5}]/u', 'A', $str );
			$str = preg_replace( '/[\x{0105}\x{0104}\x{00E0}\x{00E1}\x{00E2}\x{00E3}\x{00E4}\x{00E5}]/u', 'a', $str );
			$str = preg_replace( '/[\x{00C7}\x{0106}\x{0108}\x{010A}\x{010C}]/u', 'C', $str );
			$str = preg_replace( '/[\x{00E7}\x{0107}\x{0109}\x{010B}\x{010D}}]/u', 'c', $str );
			$str = preg_replace( '/[\x{010E}\x{0110}]/u', 'D', $str );
			$str = preg_replace( '/[\x{010F}\x{0111}]/u', 'd', $str );
			$str = preg_replace( '/[\x{00C8}\x{00C9}\x{00CA}\x{00CB}\x{0112}\x{0114}\x{0116}\x{0118}\x{011A}\x{20AC}]/u', 'E', $str );
			$str = preg_replace( '/[\x{00E8}\x{00E9}\x{00EA}\x{00EB}\x{0113}\x{0115}\x{0117}\x{0119}\x{011B}]/u', 'e', $str );
			$str = preg_replace( '/[\x{00CC}\x{00CD}\x{00CE}\x{00CF}\x{0128}\x{012A}\x{012C}\x{012E}\x{0130}]/u', 'I', $str );
			$str = preg_replace( '/[\x{00EC}\x{00ED}\x{00EE}\x{00EF}\x{0129}\x{012B}\x{012D}\x{012F}\x{0131}]/u', 'i', $str );
			$str = preg_replace( '/[\x{0142}\x{0141}\x{013E}\x{013A}]/u', 'l', $str );
			$str = preg_replace( '/[\x{00F1}\x{0148}]/u', 'n', $str );
			$str = preg_replace( '/[\x{00D2}\x{00D3}\x{00D4}\x{00D5}\x{00D6}\x{00D8}]/u', 'O', $str );
			$str = preg_replace( '/[\x{00F2}\x{00F3}\x{00F4}\x{00F5}\x{00F6}\x{00F8}]/u', 'o', $str );
			$str = preg_replace( '/[\x{0159}\x{0155}]/u', 'r', $str );
			$str = preg_replace( '/[\x{015B}\x{015A}\x{0161}]/u', 's', $str );
			$str = preg_replace( '/[\x{00DF}]/u', 'ss', $str );
			$str = preg_replace( '/[\x{0165}]/u', 't', $str );
			$str = preg_replace( '/[\x{00D9}\x{00DA}\x{00DB}\x{00DC}\x{016E}\x{0170}\x{0172}]/u', 'U', $str );
			$str = preg_replace( '/[\x{00F9}\x{00FA}\x{00FB}\x{00FC}\x{016F}\x{0171}\x{0173}]/u', 'u', $str );
			$str = preg_replace( '/[\x{00FD}\x{00FF}]/u', 'y', $str );
			$str = preg_replace( '/[\x{017C}\x{017A}\x{017B}\x{0179}\x{017E}]/u', 'z', $str );
			$str = preg_replace( '/[\x{00C6}]/u', 'AE', $str );
			$str = preg_replace( '/[\x{00E6}]/u', 'ae', $str );
			$str = preg_replace( '/[\x{0152}]/u', 'OE', $str );
			$str = preg_replace( '/[\x{0153}]/u', 'oe', $str );
			$str = preg_replace( '/[\x{2105}]/u', 'c/o', $str );
			$str = preg_replace( '/[\x{2116}]/u', 'No', $str );
			$str = preg_replace( '/[\x{0022}\x{0025}\x{0026}\x{0027}\x{00A1}\x{00A2}\x{00A3}\x{00A4}\x{00A5}\x{00A6}\x{00A7}\x{00A8}\x{00AA}\x{00AB}\x{00AC}\x{00AD}\x{00AE}\x{00AF}\x{00B0}\x{00B1}\x{00B2}\x{00B3}\x{00B4}\x{00B5}\x{00B6}\x{00B7}\x{00B8}\x{00BA}\x{00BB}\x{00BC}\x{00BD}\x{00BE}\x{00BF}\x{2019}]/u', ' ', $str );

			return $str;
		}

		/**
		 * Get parcel weight for the export DAT to decagram
		 *
		 * @param string $weight_unit   : weight unit based on the shop store
		 * @param float $weight         : parcel weight
		 *
		 * @return float                : return weight in decagram
		 */
		public static function get_converted_parcel_weight_for_export( $weight_unit, $weight ) {
			if ( $weight_unit === 'kg' ) {
				return $weight * 100;
			}
			if ( $weight_unit === 'g' ) {
				return $weight / 10;
			}
			if ( $weight_unit === 'lbs' ) {
				return $weight * 45.3592;
			}
			if ( $weight_unit === 'oz' ) {
				return $weight * 2.83495;
			}
		}

		/**
		 * Get parcel weight for dpd delivery options settings, registered in kg
		 *
		 * @param string $weight_unit   : weight unit in kg
		 * @param float $weight         : parcel weight
		 *
		 * @return float
		 */
		public static function get_converted_parcel_weight_dpd_setting( $weight_unit, $weight ) {
			if ( $weight_unit === 'kg' ) {
				return $weight;
			}
			if ( $weight_unit === 'g' ) {
				return $weight * 1000;
			}
			if ( $weight_unit === 'lbs' ) {
				return $weight / 0.45359237;
			}
			if ( $weight_unit === 'oz' ) {
				return $weight / 0.0283495231;
			}
		}

		/**
		 * Get company name, return value depends of the company field status
		 *
		 * @param $order_data
		 *
		 * @return mixed
		 */
		public static function get_company_name( $order_data ) {
			if ( $order_data['order_shipping_method_id'] === 'dpdfrance_relais' && empty( $order_data['shipping_company'] ) ) {
				return $order_data['shipping_address_2'];
			}

			return $order_data['shipping_company'];
		}

		/**
		 * Get relais Id ex: PXXXXX
		 *
		 * @param $company_value
		 *
		 * @return false|string
		 */
		public static function get_relais_id( $company_value ) {
			return substr( $company_value, -7, 6 );
		}

		/**
		 * Get the total weight of the order
		 *
		 * @param $order
		 *
		 * @return float|int|WP_Error
		 */
		public static function get_order_total_weight( $order ) {
			if ( !$order ) {
				return new WP_Error( 'broke', __( "There are no order", "woocommerce-dpdfrance" ) );
			}
			$order_weight = 0;
			foreach ( $order->get_items() as $item ) {
				if ( $item['qty'] > 0 ) {
					if ( self::is_from_wc3() ) {
						$variation = wc_get_product( $item->get_variation_id() );
						$product   = $variation ?: wc_get_product( $item['product_id'] );
						if ( $product ) {
							$order_weight += ( (float) $product->get_weight() * $item['qty'] );
						}
					} else {
						$order_weight += ( (float) $order->get_product_from_item( $item )->get_weight() * $item['qty'] );
					}
				}
			}

			return $order_weight;
		}

		/**
		 * Get shipper code depending of the order shipping method
		 *
		 * @param $order_infos
		 *
		 * @return mixed
		 */
		public static function get_shipper_code( $order_infos ) {
			$dpdrelais_shipper_data  = get_option( 'woocommerce_dpdfrance_relais_settings' );
			$dpdpredict_shipper_data = get_option( 'woocommerce_dpdfrance_predict_settings' );
			$dpdworld_shipper_data   = get_option( 'woocommerce_dpdfrance_world_settings' );
			$dpdclassic_shipper_data = get_option( 'woocommerce_dpdfrance_classic_settings' );
			if ( is_array( $order_infos ) ) {
				if ( $order_infos['order_shipping_method_id'] === 'dpdfrance_relais' || preg_match( '/\(P\d{5}\)/', $order_infos['shipping_company'] ) ) {
					return $dpdrelais_shipper_data['shipper_code'];
				}

				if ( $order_infos['order_shipping_method_id'] === 'dpdfrance_predict' && !empty( $order_infos['customer_phone'] ) ) {
					return $dpdpredict_shipper_data['shipper_code'];
				}

				if ( $order_infos['order_shipping_method_id'] === 'dpdfrance_world' && !empty( $order_infos['customer_phone'] ) ) {
					return $dpdworld_shipper_data['shipper_code'];
				}
			} else {
				$shipping_method_id = self::get_order_shipping_method_id( $order_infos );
				if ( $shipping_method_id === 'dpdfrance_relais' ) {
					return $dpdrelais_shipper_data['shipper_code'];
				}

				if ( $shipping_method_id === 'dpdfrance_predict' ) {
					return $dpdpredict_shipper_data['shipper_code'];
				}

				if ( $shipping_method_id === 'dpdfrance_world' ) {
					return $dpdworld_shipper_data['shipper_code'];
				}
			}

			return $dpdclassic_shipper_data['shipper_code'];
		}

		/**
		 * Get depot code depending of the order shipping method
		 *
		 * @param $order_infos
		 *
		 * @return mixed
		 */
		public static function get_depot_code( $order_infos ) {
			$dpdrelais_shipper_data  = get_option( 'woocommerce_dpdfrance_relais_settings' );
			$dpdpredict_shipper_data = get_option( 'woocommerce_dpdfrance_predict_settings' );
			$dpdworld_shipper_data   = get_option( 'woocommerce_dpdfrance_world_settings' );
			$dpdclassic_shipper_data = get_option( 'woocommerce_dpdfrance_classic_settings' );
			if ( is_array( $order_infos ) ) {
				if ( $order_infos['order_shipping_method_id'] === 'dpdfrance_relais' || preg_match( '/\(P\d{5}\)/', $order_infos['shipping_company'] ) ) {
					return $dpdrelais_shipper_data['depot_code'];
				}

				if ( $order_infos['order_shipping_method_id'] === 'dpdfrance_predict' && !empty( $order_infos['customer_phone'] ) ) {
					return $dpdpredict_shipper_data['depot_code'];
				}

				if ( $order_infos['order_shipping_method_id'] === 'dpdfrance_world' && !empty( $order_infos['customer_phone'] ) ) {
					return $dpdworld_shipper_data['depot_code'];
				}
			} else {
				$shipping_method_id = self::get_order_shipping_method_id( $order_infos );
				if ( $shipping_method_id === 'dpdfrance_relais' ) {
					return $dpdrelais_shipper_data['depot_code'];
				}

				if ( $shipping_method_id === 'dpdfrance_predict' ) {
					return $dpdpredict_shipper_data['depot_code'];
				}

				if ( $shipping_method_id === 'dpdfrance_world' ) {
					return $dpdworld_shipper_data['depot_code'];
				}
			}


			return $dpdclassic_shipper_data['depot_code'];
		}

		/**
		 * Get shipping method icon depending of the order shipping method
		 *
		 * @param $order_infos
		 *
		 * @return mixed
		 */
		public static function get_shipping_method_icon( $order_infos ) {
			if ( $order_infos['order_shipping_method_id'] === 'dpdfrance_relais' || preg_match( '/\(P\d{5}\)/', $order_infos['shipping_company'] ) ) {
				return '<img src="' . OS_DPDFRANCE_ROOT_URL . '/assets/img/admin/service_relais.png" alt="Relais" title="Relais"/>';
			}

			if ( $order_infos['order_shipping_method_id'] === 'dpdfrance_predict' && !empty( $order_infos['customer_phone'] ) ) {
				return '<img src="' . OS_DPDFRANCE_ROOT_URL . '/assets/img/admin/service_predict.png" alt="Predict" title="Predict"/>';
			}

			if ( $order_infos['order_shipping_method_id'] === 'dpdfrance_world' && !empty( $order_infos['customer_phone'] ) ) {
				return '<img src="' . OS_DPDFRANCE_ROOT_URL . '/assets/img/admin/service_world.png" alt="Intercontinental" title="Intercontinental"/>';
			}

			return '<img src="' . OS_DPDFRANCE_ROOT_URL . '/assets/img/admin/service_dom.png" alt="Classic" title="Classic"/>';
		}

		/**
		 * Return address link to a map or a tracking link
		 *
		 * @param $order_infos
		 *
		 * @return string
		 */
		public static function get_address_link( $order_infos ): string {
			$order_company       = self::get_company_name( $order_infos );
			$google_map_base_url = 'https://maps.google.com/maps?f=q&hl=fr&geocode=&q=';
			$google_map_address  = str_replace( ' ', '+', $order_infos['shipping_address_1'] ) . ',' . str_replace( ' ', '+', $order_infos['shipping_postcode'] ) . '+' . str_replace( ' ', '+', $order_infos['shipping_city'] );
			$address             = $order_infos['order_shipping_method_id'] === 'dpdfrance_relais' ?
				$order_company . '<br/>' . $order_infos['shipping_address_1'] . '<br/>' . $order_infos['shipping_postcode'] . ' ' . $order_infos['shipping_city'] :
				( $order_infos['shipping_company'] ? strtoupper( $order_infos['shipping_company'] ) . '<br/>' : '' ) . strtoupper( $order_infos['shipping_address_1'] ) . '<br/>' . $order_infos['shipping_postcode'] . ' ' . strtoupper( $order_infos['shipping_city'] );

			if ( $order_infos['order_shipping_method_id'] === 'dpdfrance_relais' || preg_match( '/\(P\d{5}\)/', $order_company ) ) {
				return '<a class="popup" href="https://www.dpd.fr/home/shipping/relais_details.php?idPR=' . self::get_relais_id( $order_company ) . '" target="_blank">' . $address . '</a>';
			}

			if ( $order_infos['order_shipping_method_id'] === 'dpdfrance_predict' && !empty( $order_infos['customer_phone'] ) ) {
				return "<a class='popup' href='" . $google_map_base_url . $google_map_address . "&output=embed' target='_blank'>" . $address . "</a>";
			}

			if ( $order_infos['order_shipping_method_id'] === 'dpdfrance_world' ) {
				return "<a class='popup' href='" . $google_map_base_url . $google_map_address . "&output=embed' target='_blank'>" . $address . "</a>";
			}

			return "<a class='popup' href='" . $google_map_base_url . $google_map_address . "&output=embed' target='_blank'>" . $address . "</a>";
		}

		/**
		 * Get order shipping method id
		 *
		 * @param $order
		 *
		 * @return mixed|string
		 */
		public static function get_order_shipping_method_id( $order ) {
			$order_shipping_method_id = '';
			foreach ( $order->get_shipping_methods() as $shipping_method ) {
				$order_shipping_method_id = $shipping_method['method_id'];
			}

			return $order_shipping_method_id;
		}

		/**
		 * Get Iso Code
		 * if iso code is european then convert it to DPD Station format
		 * else convert to intercontinental format
		 *
		 * @param $order_infos
		 *
		 * @return array|string|string[]
		 */
		public static function get_iso_code( $order_infos ) {
			$iso_standard = WC_DPDFrance_Constants::ISO_STANDARD;
			$iso_eprint   = WC_DPDFrance_Constants::ISO_EPRINT;

			return in_array( $order_infos['shipping_country'], $iso_standard, true ) ?
				str_replace( $iso_standard, $iso_eprint, $order_infos['shipping_country'] ) :
				str_replace( $order_infos['shipping_country'], "INT", $order_infos['shipping_country'] );
		}

		/**
		 * Retrieve orders ID except delivered and completed statuses
		 * @return array
		 */
		public static function get_orders_id_except_delivered_and_completed_status(): array {
			global $wpdb;

			$sql = "SELECT      ID 
                	FROM        {$wpdb->posts} 
                	WHERE       post_type = 'shop_order' 
                	AND         post_status NOT IN ('wc-completed' , 'wc-cancelled' , 'wc-failed' , 'wc-refunded' , 'wc-closed' , '" . get_option( 'wc_settings_tab_dpdfrance_etape_livre' ) . "') 
                    ORDER BY    id DESC 
                    LIMIT       2000 
                ";

			return $wpdb->get_col( $sql );
		}

		/**
		 * Get order shipping cost which might include if islands zone or mountain field are not empty
		 *
		 * @param $shipping_method
		 * @param $postcode
		 * @param $cost
		 *
		 * @return array
		 */
		public static function get_order_shipping_cost( $shipping_method, $postcode, $cost = 0 ): array {
			if ( !in_array( $shipping_method, WC_DPDFrance_Constants::DPD_SHIPPING_METHODS, true ) ) {
				wp_die( 'This is not a DPD shipping methods' );
			}
			$exclude        = 0;
			$setting        = get_option( 'woocommerce_' . $shipping_method . '_settings' );
			$supp_island    = $setting['supp_iles'];
			$supp_mountain  = $setting['supp_montagne'];
			$islands_zone   = WC_DPDFrance_Constants::ISLANDS_ZONE;
			$mountains_zone = WC_DPDFrance_Constants::MOUNTAINS_ZONE;

			// * IF postcode start with 20XXX OR postcode is in zone iles
			if ( strpos( $postcode, '20' ) === 0 || in_array( $postcode, $islands_zone, true ) ) {
				if ( $supp_island >= 0 ) {
					$cost += $supp_island;
				} else {
					$exclude = WC_DPDFrance_Constants::ERROR_ISLAND_ZONE;
				}
			}

			if ( in_array( $postcode, $mountains_zone, true ) ) {
				if ( $supp_mountain >= 0 ) {
					$cost += $supp_mountain;
				} else {
					$exclude = WC_DPDFrance_Constants::ERROR_MOUNTAIN_ZONE;
				}
			}

			return [
				'excl' => $exclude,
				'cost' => $cost
			];
		}

		/**
		 * Connect to SOAP and get pudo list
		 *
		 * @param $service_url
		 * @param $variable
		 *
		 * @return string|void
		 */
		public static function get_pudo_list( $service_url, $variable ) {
			try {
				ini_set( 'default_socket_timeout', 5 );
				$soap_pudo = new SoapClient( $service_url, [
					'connection_timeout' => 5,
					'cache_wsdl'         => WSDL_CACHE_NONE,
					'exceptions'         => true
				] );

				return $soap_pudo->getPudoList( $variable )->GetPudoListResult->any;
			} catch ( Exception $e ) {
				return __( 'DPD Relais is not available at the moment, please try again shortly.', 'woocommerce-dpdfrance' );
			}
		}

		/**
		 * Prepare variable for pudo query
		 *
		 * @param $carrier
		 * @param $key
		 * @param $address
		 * @param $zipcode
		 * @param $city
		 * @param $date
		 *
		 * @return array
		 */
		public static function prepare_pudo_query_variable( $carrier, $key, $address, $zipcode, $city, $date ): array {
			return [
				'carrier'             => $carrier,
				'key'                 => $key,
				'address'             => $address,
				'zipCode'             => $zipcode,
				'city'                => $city,
				'countrycode'         => 'FR',
				'requestID'           => '1234',
				'request_id'          => '1234',
				'date_from'           => $date,
				'max_pudo_number'     => '',
				'max_distance_search' => '',
				'weight'              => '',
				'category'            => '',
				'holiday_tolerant'    => ''
			];
		}

		/**
		 * Prepare single pickup point
		 *
		 * @param $item
		 *
		 * @return array
		 */
		public static function prepare_single_pickup_point( $item ): array {
			$point              = [];
			$item               = (array) $item;
			$point['relay_id']  = $item['PUDO_ID'];
			$point['shop_name'] = self::strip_accents( $item['NAME'] );
			$point['address1']  = self::strip_accents( $item['ADDRESS1'] );
			if ( $item['ADDRESS2'] ) {
				$point['address2'] = self::strip_accents( $item['ADDRESS2'] );
			}
			if ( $item['ADDRESS3'] ) {
				$point['address3'] = self::strip_accents( $item['ADDRESS3'] );
			}
			if ( $item['LOCAL_HINT'] ) {
				$point['local_hint'] = self::strip_accents( $item['LOCAL_HINT'] );
			}
			$point['postal_code'] = $item['ZIPCODE'];
			$point['city']        = self::strip_accents( $item['CITY'] );
			$point['distance']    = number_format( $item['DISTANCE'] / 1000, 2 );
			$point['coord_lat']   = (float) str_replace( ',', '.', $item['LATITUDE'] );
			$point['coord_long']  = (float) str_replace( ',', '.', $item['LONGITUDE'] );

			$days = WC_DPDFrance_Constants::WEEKDAYS;

			if ( count( $item['OPENING_HOURS_ITEMS']->OPENING_HOURS_ITEM ) > 0 ) {
				foreach ( $item['OPENING_HOURS_ITEMS']->OPENING_HOURS_ITEM as $k => $oh_item ) {
					$oh_item                                = (array) $oh_item;
					$point[ $days[ $oh_item['DAY_ID'] ] ][] = $oh_item['START_TM'] . ' - ' . $oh_item['END_TM'];
				}
			}

			if ( empty( $point['monday'] ) ) {
				$h1 = __( 'Closed', 'woocommerce-dpdfrance' );
			} else {
				$h1 = empty( $point['monday'][1] ) ?
					$point['monday'][0] :
					$point['monday'][0] . ' & ' . $point['monday'][1];
			}

			if ( empty( $point['tuesday'] ) ) {
				$h2 = __( 'Closed', 'woocommerce-dpdfrance' );
			} else {
				$h2 = empty( $point['tuesday'][1] ) ?
					$point['tuesday'][0] :
					$point['tuesday'][0] . ' & ' . $point['tuesday'][1];
			}

			if ( empty( $point['wednesday'] ) ) {
				$h3 = __( 'Closed', 'woocommerce-dpdfrance' );
			} else {
				$h3 = empty( $point['wednesday'][1] ) ?
					$point['wednesday'][0] :
					$point['wednesday'][0] . ' & ' . $point['wednesday'][1];
			}

			if ( empty( $point['thursday'] ) ) {
				$h4 = __( 'Closed', 'woocommerce-dpdfrance' );
			} else {
				$h4 = empty( $point['thursday'][1] ) ?
					$point['thursday'][0] :
					$point['thursday'][0] . ' & ' . $point['thursday'][1];
			}

			if ( empty( $point['friday'] ) ) {
				$h5 = __( 'Closed', 'woocommerce-dpdfrance' );
			} else {
				$h5 = empty( $point['friday'][1] ) ?
					$point['friday'][0] :
					$point['friday'][0] . ' & ' . $point['friday'][1];
			}

			if ( empty( $point['saturday'] ) ) {
				$h6 = __( 'Closed', 'woocommerce-dpdfrance' );
			} else {
				$h6 = empty( $point['saturday'][1] ) ?
					$point['saturday'][0] :
					$point['saturday'][0] . ' & ' . $point['saturday'][1];
			}

			if ( empty( $point['sunday'] ) ) {
				$h7 = __( 'Closed', 'woocommerce-dpdfrance' );
			} else {
				$h7 = empty( $point['sunday'][1] ) ?
					$point['sunday'][0] :
					$point['sunday'][0] . ' & ' . $point['sunday'][1];
			}

			$point['opening_hours'] = [
				'monday'    => $h1,
				'tuesday'   => $h2,
				'wednesday' => $h3,
				'thursday'  => $h4,
				'friday'    => $h5,
				'saturday'  => $h6,
				'sunday'    => $h7
			];

			if ( count( $item['HOLIDAY_ITEMS']->HOLIDAY_ITEM ) > 0 ) {
				$x = 0;
			}
			foreach ( $item['HOLIDAY_ITEMS']->HOLIDAY_ITEM as $holiday_item ) {
				$holiday_item                  = (array) $holiday_item;
				$point['closing_period'][ $x ] = $holiday_item['START_DTM'] . ' - ' . $holiday_item['END_DTM'];
				++$x;
			}

			return $point;
		}

        /**
         * Get suggested pickup point
         * @param $pudo_list
         * @param $pudo_displayed
         * @return array
         */
		public static function get_suggested_pickup_points( $pudo_list, $pudo_displayed): array {
			$dpd_relais_points = [];
			$relais_items      = $pudo_list->PUDO_ITEMS;
			$limit_suggestions = 0;
			foreach ( $relais_items->PUDO_ITEM as $pickup_point ) {
				$conditions = self::get_order_shipping_cost( 'dpdfrance_relais', $pickup_point['ZIPCODE'] );
				if ( isset( $conditions['excl'] ) && $conditions['excl'] !== 0 ) {
					continue;
				}
				$dpd_relais_points[] = self::prepare_single_pickup_point( $pickup_point );
				if ( ++$limit_suggestions === $pudo_displayed) {
					break;
				}
			}

			return $dpd_relais_points;
		}

        /**
         * Get total cart weight
         * @param array $package : cart context
         * @return float|int     : cart weight
         */
		public static function get_cart_weight( $package ) {
			$total = 0;
			foreach ( $package['contents'] as $cart_item ) {
				$product = $cart_item['data'];
				$total   += ( (float) $product->get_weight() * $cart_item['quantity'] );
			}

			return $total;
		}

		/**
		 * Return true if value less than max, incl. "*"
		 *
		 * @param $value
		 * @param $max
		 *
		 * @return bool
		 */
		public static function is_less_than( $value, $max ): bool {
			if ( $max === '*' ) {
				return true;
			}

			return ( $value <= $max );
		}

		/**
		 * Generates dropdown menus in delivery methods settings
		 *
		 * @param $zones
		 *
		 * @return array
		 */
		public static function admin_prepare_dropdown_options( $zones ): array {
			$option_arr = [];

			/* Get registered Shipping Zones to table_rate_zone array*/
			foreach ( $zones as $option ) {
				$option_arr['table_rate_zone'][ esc_attr( $option['id'] ) ] = esc_js( $option['name'] );
			}

			/* Add Everywhere else Shipping Zones to table_rate_zone array*/
			$option_arr['table_rate_zone']['0'] = __( 'Everywhere Else', 'woocommerce-dpdfrance' );

			/* Every country to country array */
			foreach ( WC()->countries->get_shipping_countries() as $id => $value ) {
				$option_arr['country'][ esc_attr( $id ) ] = esc_js( $value );
			}

			/* Add unit basis to rate basis array*/
			$option_arr['rate_basis']['weight'] = sprintf( __( 'Weight (%s)', 'woocommerce-dpdfrance' ), get_option( 'woocommerce_weight_unit' ) );
			$option_arr['rate_basis']['price']  = sprintf( __( 'Total (%s)', 'woocommerce-dpdfrance' ), get_woocommerce_currency_symbol() );

			return $option_arr;
		}

		/**
		 * Retrieves cheapest rate from a list of table_rates.
		 *
		 * @param $table_rates
		 *
		 * @return mixed
		 */
		public static function pick_cheapest_table_rate( $table_rates ) {
			$cheapest = false;
			foreach ( $table_rates as $table_rate ) {
				if ( $cheapest === false ) {
					$cheapest = $table_rate;
				} else if ( $table_rate['cost'] < $cheapest['cost'] ) {
					$cheapest = $table_rate;
				}
			}

			return $cheapest;
		}

		/**
		 * Retrieves available zone ids for supplied shipping address
		 *
		 * @param $package
		 * @param $zones
		 *
		 * @return array
		 */
		public static function get_available_zones( $package, $zones ): array {

			$destination_country = $package['destination']['country'];
			$available_zones     = [];

			foreach ( $zones as $zone ) {
				if ( !empty( $zone['country'] ) && in_array( $destination_country, $zone['country'], true ) ) {
					$available_zones[] = $zone['id'];
				}
			}

			if ( empty( $available_zones ) ) {
				$found = false;
				foreach ( WC()->countries->get_shipping_countries() as $id => $value ) {
					if ( $destination_country == $id ) {
						$found = true;
					}
				}
				if ( $found ) {
					$available_zones[] = '0'; // "Everywhere else" zone
				}
			}

			return $available_zones;
		}

		/**
		 * Get cart taxes
		 * @return float|int
		 */
		public static function get_cart_taxes() {
			global $woocommerce;

			return WC()->version >= '3.2.0' ?
				array_sum( $woocommerce->cart->get_cart_contents_taxes() ) :
				array_sum( $woocommerce->cart->taxes );
		}

		/**
		 * Return true if user table rate is in the available zone
		 *
		 * @param $table_rate
		 * @param $available_zones
		 *
		 * @return bool
		 */
		public static function check_table_rate_in_available_zone( $table_rate, $available_zones ): bool {
			return in_array( $table_rate, $available_zones, true );
		}

		/**
		 * Return true if cart weight is valid for the shipping
		 *
		 * @param array $table_rate : DPD Settings table rates
		 * @param float $weight     : Cart weight
		 *
		 * @return bool
		 */
		public static function check_cart_weight_is_valid_for_shipping( $table_rate, $weight ): bool {
			if ( $table_rate['basis'] === 'weight' ) {
				return ( ( $weight >= $table_rate['min'] ) && ( self::is_less_than( $weight, $table_rate['max'] ) ) );
			}

			return true;
		}

		/**
		 * Return true if cart weight is valid for the shipping
		 *
		 * @param $table_rate
		 * @param $weight
		 *
		 * @return bool
		 */
		public static function check_cart_total_price_is_valid_for_shipping( $cart_total_price, $table_rate ): bool {
			if ( $table_rate['basis'] === 'price' ) {
				return ( ( $cart_total_price >= $table_rate['min'] ) && ( self::is_less_than( $cart_total_price, $table_rate['max'] ) ) );
			}

			return true;
		}

		/**
		 * Retrieves available table_rates for cart and supplied shipping address
		 *
		 * @param array $package        : cart context
		 * @param array $zones          : DPD delivery zones
		 * @param array $table_rates    : DPD table rate of the delivery zones
		 *
		 * @return array
		 */
		public static function get_available_table_rates( $package, $zones, $table_rates ): array {
			global $woocommerce;

			$available_table_rates = [];
			$available_zones       = self::get_available_zones( $package, $zones );
			$weight                = self::get_cart_weight( $package );
			$cart_taxes            = self::get_cart_taxes();
			$cart_total_price      = $woocommerce->cart->cart_contents_total + $cart_taxes;

			foreach ( $table_rates as $table_rate ) {
				$zone_pass   = self::check_table_rate_in_available_zone( $table_rate['zone'], $available_zones );
				$weight_pass = self::check_cart_weight_is_valid_for_shipping( $table_rate, $weight );
				$total_pass  = self::check_cart_total_price_is_valid_for_shipping( $cart_total_price, $table_rate );

				// Accept table_rate if passes all tests
				if ( $zone_pass && $weight_pass && $total_pass ) {
					$available_table_rates[] = $table_rate;
				}
			}

			return $available_table_rates;
		}

		/**
		 * Check if cart has handle fee
		 *
		 * @param $settings
		 * @param $table_rate
		 *
		 * @return mixed
		 */
		public static function check_cart_has_handle_fee( $settings, $table_rate ) {
			if ( !empty( $settings['handling_fee'] ) && $settings['handling_fee'] > 0 ) {
				return $table_rate['cost'] + $settings['handling_fee'];
			}

			return $table_rate['cost'];
		}

		/**
		 * Check if cart can use coupons
		 *
		 * @param $settings
		 * @param $cost_condition
		 *
		 * @return mixed
		 */
		public static function check_cart_can_use_coupons( $settings, $cost_condition ) {
			$free_shipping_cost = 0;

			if ( ( $settings['coupon_freeshipping'] === 'enabled' ) && $coupons = WC()->cart->get_coupons() ) {
				foreach ( $coupons as $coupon ) {
					if ( $coupon->is_valid() && $coupon->get_free_shipping() ) {
						return $free_shipping_cost;
					}
				}
			}

			return $cost_condition;
		}

		/**
		 * Check if cart can have free shipping
		 *
		 * @param $settings
		 * @param $cost_condition
		 *
		 * @return mixed
		 */
		public static function check_cart_can_have_free_shipping( $settings, $cost_condition ) {
			global $woocommerce;

			$free_shipping_cost = 0;
			if ( !empty( $settings['franco'] ) && $settings['franco'] > 0 ) {
				$cart_taxes = self::get_cart_taxes();
				$total      = $woocommerce->cart->cart_contents_total + $cart_taxes;
				if ( $total >= $settings['franco'] ) {
					return $free_shipping_cost;
				}
			}

			return $cost_condition;
		}

		/**
		 * Check if cart is overweight
		 *
		 * @param array $package    : cart context
		 * @param array $settings   : dpd cart settings related to the order
		 * @param int $conditions   : exclusion condition
		 *
		 * @return int
		 */
		public static function check_cart_is_overweighted( $package, $settings, $conditions ) {
			if ( ! empty( $settings['max_weight'] ) && ( (float) $settings['max_weight'] > 0 ) ) {
				$parcel_weight = self::get_cart_weight( $package );
				if ( $parcel_weight > ( (float) $settings['max_weight'] ) ) {
					return WC_DPDFrance_Constants::OVERWEIGHT;
				}
			}

			return $conditions;
		}

		/**
		 * Check if cart has excluded product
		 *
		 * @param $package
		 * @param $settings
		 * @param $excluded_conditions
		 *
		 * @return int|mixed
		 */
		public static function check_cart_has_excluded_products( $package, $settings, $excluded_conditions ) {
			$cart_shipping_classes = self::dpdfrance_find_shipping_classes( $package );
			$classes_to_exclude    = explode( ',', $settings['classes_to_exclude'] );

			if ( !empty( $settings['classes_to_exclude'] ) ) {

				$common_class         = array_intersect( $classes_to_exclude, $cart_shipping_classes );
				$has_class_to_exclude = false;

				foreach ( $common_class as $class ) {
					if ( $class !== '' ) {
						$has_class_to_exclude = true;
					}
				}
				if ( $has_class_to_exclude ) {
					return WC_DPDFrance_Constants::HAS_EXCLUDED_PRODUCT;
				}
			}

			return $excluded_conditions;
		}

	}
}
